<?php
namespace WooCommerce_Contact_for_Shipping_Quote;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly


/**
 * Add shipping method description.
 *
 * Add a additional description to the shipping method for the 'Contact' link.
 *
 * @since 1.0.0
 *
 * @param \WC_Shipping_Rate $rate
 * @param $index
 */
function add_shipping_method_contact_description( $rate, $index ) {
	if ( $rate->method_id != 'custom_shipping_quote' ) {
		return;
	}

	/** @var WCCSQ_Shipping_Quote_Method $method */
	$method = \WC_Shipping_Zones::get_shipping_method( $rate->get_instance_id() );
	$quote  = $method->get_quote();

	$classes  = $quote !== false ? 'wccsq-quote-requested' : '';
	$classes .= $quote && ! is_null( $quote->get_quote_amount() ) ? ' wccsq-quote-available' : '';
	?><p class="wccsq-quote-description <?php echo esc_attr( $classes ); ?>"><?php echo wp_kses_post( $method->get_description() ); ?></p><?php
}
add_filter( 'woocommerce_after_shipping_rate', '\WooCommerce_Contact_for_Shipping_Quote\add_shipping_method_contact_description', 10, 2 );


/**
 * Hide rates when Shipping Quote is available.
 *
 * Hide other rates (based on user settings) when the shipping quote
 * option is available.
 *
 * @since 1.0.0
 *
 * @param array $rates List of existing rates.
 * @param array $package Package data.
 * @return mixed List of filtered rates.
 */
function hide_rates_when_shipping_quote_is_available(  $rates, $package ) {

	$excluded_rates = array();
	$method_ids = wp_list_pluck( $rates, 'method_id' );

	if ( in_array( 'custom_shipping_quote', $method_ids ) ) {
		$rate_index = array_search( 'custom_shipping_quote', $method_ids );
		$rate = $rates[ $rate_index ];
		$shipping_method = \WC_Shipping_Zones::get_shipping_method( $rate->get_instance_id() );

		$excluded_rates = $shipping_method->get_option( 'exclude_methods' );
	}

	if ( ! empty( $excluded_rates ) ) {
		foreach ( $rates as $k => $rate ) {
			if ( array_intersect( array( $rate->id, $rate->method_id, $rate->instance_id ), $excluded_rates ) ) {
				unset( $rates[ $k ] );
			}
		}
	}

	return $rates;
}
add_filter( 'woocommerce_package_rates', '\WooCommerce_Contact_for_Shipping_Quote\hide_rates_when_shipping_quote_is_available', 10, 2 );


/**
 * Add time to package.
 *
 * Add a timestamp to the package when the Contact for Shipping Quote option has been selected.
 * This ensures shipping rates (quotes) are re-calculated on refresh to ensure updates/provided
 * quotes are pushed through.
 *
 * @since 1.1.0
 *
 * @param  array $packages Existing packages.
 * @return mixed           Modified packages.
 */
function add_package_shipping_quote_time( $packages ) {
	$chosen_methods = WC()->session->get( 'chosen_shipping_methods' );

	foreach ( $packages as $i => $package ) {

		if ( isset( $chosen_methods[ $i ] ) && strpos( $chosen_methods[ $i ], 'custom_shipping_quote' ) === 0 ) {
			$packages[ $i ]['shipping_quote_time'] = time();
		}
	}

	return $packages;
}
add_filter( 'woocommerce_cart_shipping_packages', '\WooCommerce_Contact_for_Shipping_Quote\add_package_shipping_quote_time' );


/**
 * Hide $0 cost label.
 *
 * Change the shipping label to hide the $0 label when the quote is yet to be made.
 *
 * @since 1.0.0
 *
 * @param string $label Existing label.
 * @param \WC_Shipping_Rate $method Shipping method being displayed.
 * @return string Modified label.
 */
function hide_empty_shipping_cost( $label, $method ) {
	if ( $method->method_id === 'custom_shipping_quote' && $method->cost == 0 ) {
		$label = $method->get_label();
	}

	return $label;
}
add_filter( 'woocommerce_cart_shipping_method_full_label', '\WooCommerce_Contact_for_Shipping_Quote\hide_empty_shipping_cost', 10, 2 );


/**
 * Update quote status.
 *
 * Possibly update the quote status when a order is created. When a order is created the quote is considered 'completed'.
 *
 * @since 1.0.0
 *
 * @param int $order_id ID of the order.
 * @param $data
 */
function maybe_update_shipping_quote_status( $order_id, $data ) {
	$order = wc_get_order( $order_id );
	$shipping_methods = $order->get_shipping_methods();

	/** @var \WC_Order_Item_Shipping $shipping_order_item */
	foreach ( $shipping_methods as $shipping_order_item ) {
		if ( 'custom_shipping_quote' == $shipping_order_item->get_method_id() ) {
			$quote = Shipping_Quote::read_by( array( 'address_hash' => get_address_hash(), 'cart_hash' => get_cart_hash() ), array( '%s', '%s' ) );
			if ( $quote ) {
				$quote->set_order_id( $order->get_id() );
				$quote->set_status( 'completed' );
				$quote->save();
			}
		}
	}

}
add_action( 'woocommerce_checkout_update_order_meta', '\WooCommerce_Contact_for_Shipping_Quote\maybe_update_shipping_quote_status', 10, 2 );


/**
 * Prevent completion without a quote.
 *
 * Prevent the order being completed when the quote rate is selected, but not quote is set.
 *
 * @since 1.0.0
 */
function validate_checkout_quote() {
	$chosen_methods = WC()->session->get( 'chosen_shipping_methods' );
	$packages = WC()->shipping()->get_packages();

	$has_shipping_quote = false;
	foreach ( $chosen_methods as $method ) {
		if ( strpos( $method, 'custom_shipping_quote' ) !== false ) {
			$has_shipping_quote = true;
		}
	}

	if ( ! $has_shipping_quote ) {
		return false;
	}

	foreach ( $packages as $package_key => $package ) {
		/** @var \WC_Shipping_Rate $chosen_method */
		$chosen_method = $chosen_methods[ $package_key ] ?? false;
		$chosen_method = $package['rates'][ $chosen_method ] ?? false;

		if ( $chosen_method ) {
			if ( $chosen_method->get_method_id() == 'custom_shipping_quote' ) {
				$quote = Shipping_Quote::read_by( array( 'address_hash' => get_address_hash(), 'cart_hash' => get_cart_hash() ), array( '%s', '%s' ) );

				if ( ! $quote || $quote->get_status_slug() !== 'pending' ) { // Quote must exist + have 'pending' status (only status that is valid for continuing)
					$text = apply_filters( 'WCCSQ/quote_not_available_error',
						__( 'Your shipping quote is not yet available.', 'woocommerce-contact-for-shipping-quote' ),
						$package,
						$quote
					);

					$text = apply_filters( 'woocommerce_contact_for_shipping_quote/quote_not_available_error', $text, $package, $quote ); // @deprecated

					wc_add_notice( $text, 'error' );
				}
			}
		}
	}
}
add_action( 'woocommerce_after_checkout_validation', '\WooCommerce_Contact_for_Shipping_Quote\validate_checkout_quote' );


/**
 * Output popup.
 *
 * Output the popup HTML.
 *
 * @since 1.0.0
 *
 * @param int $page_id ID of the page content to display in the popup.
 */
function popup( $page_id ) {
	$post = get_post( $page_id );
	$content = $post->post_content;
	$content = apply_filters('the_content', $content);
	$content = str_replace(']]>', ']]&gt;', $content);

	?><div class="wccsq-popup-bg" style="display: none;"></div>
	<div class="wccsq-popup" style="display: none;">
		<span class="wccsq-close"></span>
		<div class="wccsq-body">
			<?php echo wp_kses_post( $content ); ?>
		</div>
	</div><?php
}


/**
 * Add popup HTML.
 *
 * Add additional popup HTML to the page when
 *
 * @since 1.0.0
 *
 * @param \WC_Shipping_Method $method Shipping method instance.
 * @param array $rate Rate details.
 */
function add_popup_html( $rate ) {
	if ( ! is_checkout() && ! is_cart() ) return;
	if ( $rate->method_id !== 'custom_shipping_quote' ) return;

	$method = new WCCSQ_Shipping_Quote_Method( $rate->instance_id );
	$page_id = $method->get_option( 'contact_popup_page_id' );

	if ( $page_id ) {
		add_action( 'wp_footer', function() use ( $page_id ) {
			popup( $page_id );
		});
	}
}
add_action( 'woocommerce_after_shipping_rate', '\WooCommerce_Contact_for_Shipping_Quote\add_popup_html', 10, 2 );


/**
 * Email address field update on change.
 *
 * Make sure the order totals are updated on email address change when the email
 * address is a required quote field.
 *
 * @since 1.1.0
 *
 * @param  array $fields List of billing checkout fields.
 * @return array         List of modified billing checkout fields.
 */
function add_billing_email_field_class( $fields ) {
	$required_quote_fields = get_option( 'shipping_quote_required_data', array() );

	if ( in_array( 'email', $required_quote_fields ) && isset( $fields['billing_email'] ) ) {
		$fields['billing_email']['class'][] = 'update_totals_on_change';
	}

	return $fields;
}
add_filter( 'woocommerce_billing_fields', '\WooCommerce_Contact_for_Shipping_Quote\add_billing_email_field_class' );
